<?php
require_once __DIR__.'/../config/database.php';
require_once __DIR__.'/../models/FiturProdi.php';
require_once __DIR__.'/../core/KMeans.php';
require_once __DIR__.'/../core/C45.php'; // atau C45Rules

$fiturModel = new FiturProdi($pdo);

// hitung ulang fitur semua prodi setiap buka halaman (realtime dari data)
$fiturModel->hitungSemuaFitur();

// ambil data fitur
$dataFitur = $fiturModel->allWithProdi();

// siapkan data untuk K-Means
$dataK = [];
foreach ($dataFitur as $row) {
    $dataK[] = [
        $row['kode_prodi'],
        (float)$row['rata_pendaftar_5th'],
        (float)$row['rata_rasio_5th'],
        (float)$row['tren_pendaftar']
    ];
}

// jalankan K-Means
$clusters = KMeans::cluster($dataK, 3);

// simpan cluster ke DB + hitung rekomendasi kuota via C4.5 rules
foreach ($dataFitur as $row) {
    $kode = $row['kode_prodi'];
    $cluster = isset($clusters[$kode]) ? $clusters[$kode] : null;

    $rek = C45Rules::rekomendasi($row['rata_rasio_5th'], $row['tren_pendaftar']);

    $stmt = $pdo->prepare("UPDATE fitur_prodi SET cluster_kmeans=?, rekomendasi_kuota=? WHERE kode_prodi=?");
    $stmt->execute([$cluster, $rek, $kode]);
}

$dataFitur = $fiturModel->allWithProdi(); // refresh
include __DIR__.'/../templates/header.php';
?>
<div class="container mt-4">
  <h3>Analisis Peminatan & Rekomendasi Kuota</h3>
  <p>Perhitungan realtime berdasarkan data kuota & pendaftar di database.</p>

  <table class="table table-sm table-bordered">
    <thead class="table-light">
      <tr>
        <th>Kode</th>
        <th>Program Studi</th>
        <th>Rata Pendaftar</th>
        <th>Rasio 5 Th</th>
        <th>Tren</th>
        <th>Cluster K-Means</th>
        <th>Kategori Peminatan</th>
        <th>Rekomendasi Kuota (C4.5)</th>
      </tr>
    </thead>
    <tbody>
      <?php foreach ($dataFitur as $row): ?>
      <tr>
        <td><?= htmlspecialchars($row['kode_prodi']) ?></td>
        <td><?= htmlspecialchars($row['nama_prodi']) ?></td>
        <td><?= number_format($row['rata_pendaftar_5th'], 0) ?></td>
        <td><?= number_format($row['rata_rasio_5th'], 2) ?></td>
        <td><?= number_format($row['tren_pendaftar'], 2) ?></td>
        <td><?= $row['cluster_kmeans'] ?></td>
        <td><?= htmlspecialchars($row['kategori_peminatan']) ?></td>
        <td>
          <?php
            $rk = $row['rekomendasi_kuota'];
            if ($rk == 'Naik')   echo '<span class="badge bg-success">Naik</span>';
            elseif ($rk == 'Turun') echo '<span class="badge bg-danger">Turun</span>';
            else                 echo '<span class="badge bg-warning text-dark">Tetap</span>';
          ?>
        </td>
      </tr>
      <?php endforeach; ?>
    </tbody>
  </table>
</div>
<?php
// siapkan data untuk grafik
$labels = [];
$rataPendaftar = [];
$clustersBar = [];
$rekLabels = ['Naik' => 0, 'Tetap' => 0, 'Turun' => 0];

foreach ($dataFitur as $row) {
    $labels[] = $row['nama_prodi'];
    $rataPendaftar[] = (int)$row['rata_pendaftar_5th'];
    $clustersBar[] = (int)$row['cluster_kmeans'];
    $rk = $row['rekomendasi_kuota'];
    if (isset($rekLabels[$rk])) {
        $rekLabels[$rk]++;
    }
}
?>
<div class="row mt-4">
  <div class="col-md-7">
    <h5>Grafik Rata-Rata Pendaftar per Prodi</h5>
    <canvas id="chartPendaftar" height="120"></canvas>
  </div>
  <div class="col-md-5">
    <h5>Distribusi Rekomendasi Kuota</h5>
    <canvas id="chartRekom" height="120"></canvas>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
const labels = <?= json_encode($labels) ?>;
const dataPendaftar = <?= json_encode($rataPendaftar) ?>;
const clusters = <?= json_encode($clustersBar) ?>;

// warna berdasarkan cluster (1,2,3)
const colors = clusters.map(c => {
  if (c === 1) return 'rgba(54, 162, 235, 0.7)';     // biru
  if (c === 2) return 'rgba(75, 192, 192, 0.7)';     // hijau
  return 'rgba(255, 159, 64, 0.7)';                  // oranye
});

const ctx1 = document.getElementById('chartPendaftar').getContext('2d');
new Chart(ctx1, {
  type: 'bar',
  data: {
    labels: labels,
    datasets: [{
      label: 'Rata-rata Pendaftar',
      data: dataPendaftar,
      backgroundColor: colors,
      borderWidth: 1
    }]
  },
  options: {
    responsive: true,
    plugins: {
      legend: { display: false }
    },
    scales: {
      y: { beginAtZero: true, title: { display: true, text: 'Jumlah Pendaftar' } },
      x: { title: { display: true, text: 'Program Studi' } }
    }
  }
});

const rekCounts = <?= json_encode(array_values($rekLabels)) ?>;
const rekNames  = <?= json_encode(array_keys($rekLabels)) ?>;
const ctx2 = document.getElementById('chartRekom').getContext('2d');
new Chart(ctx2, {
  type: 'pie',
  data: {
    labels: rekNames,
    datasets: [{
      data: rekCounts,
      backgroundColor: [
        'rgba(40, 167, 69, 0.7)',  // Naik
        'rgba(255, 193, 7, 0.7)',  // Tetap
        'rgba(220, 53, 69, 0.7)'   // Turun
      ]
    }]
  },
  options: {
    responsive: true,
    plugins: {
      legend: { position: 'bottom' }
    }
  }
});
</script>
<?php include __DIR__.'/../templates/footer.php'; ?>
